import calendar
import os
import shutil
import time
import warnings
import pandas as pd
from ydata_profiling import ProfileReport
from tabulate import tabulate
from inputdata import InputData
from mltool import MlTool
from validation import run_uncertainty
import joblib


def main():
    """
    Run MLTool and Hybrid model calculations
    :return: None
    """
    # setting output configurations
    warnings.filterwarnings("ignore")
    pd.set_option("display.precision", 2)
    pd.set_option("display.float_format", lambda x: "%.2f" % x)
    input_data = InputData('parameters.yaml')
    hybrid_models = pd.DataFrame()

    if os.path.isdir('output'):
        shutil.move('output', 'output-' + str(calendar.timegm(time.gmtime())))
    os.mkdir('output')
    
    if input_data.user_param['pandas_profiling']:
        profile = ProfileReport(pd.read_csv(input_data.user_param['datafile']), 
                                title="Pandas Profiling Report")
        profile.to_file("output/pandas_profiling.html")

    for i in range(0, input_data.user_param['num_hybrid_model']):
        reg = MlTool()
        scores, predictions, predictions_all, models = reg.fit(input_data)
        print(tabulate(scores, headers="keys", tablefmt='psql', showindex=True))
        if input_data.user_param['save_output']:
            scores.to_csv('output/scores_' + str(i) + '.csv')
        rm = input_data.user_param['ranking_metric']

        if rm in ["Adjusted R-Squared", "R-Squared"]:
            best_model = scores[scores[rm] == scores[rm].max()].head(1)

        else:
            best_model = scores[scores[rm] == scores[rm].min()].head(1)

        regress = [est for est in input_data.regress if
                   (best_model.index[0] == est[0])][0]

        if input_data.user_param['uncertainty_calculation']['active']:
            uncern, alpha = run_uncertainty(regress, input_data)
            uncern.to_csv('output/uncertainties_' + str(i) + '_cf=' + str(alpha) + '.csv', index=False)

        if input_data.user_param['save_output']:
            joblib.dump(models[best_model.index.values[0]],
                        'output/' + best_model.index.values[0] + '-' + str(i) + '.pkl',
                        compress=1)

        hybrid_models = hybrid_models.append(best_model)
        hyb_y = (predictions_all[best_model.index].values - pd.DataFrame(input_data.y).values).flatten()
        input_data.y = hyb_y
        input_data.set_train_test()

    print(tabulate(hybrid_models, headers="keys", tablefmt='psql', showindex=True))


main()
