import numpy as np
from sklearn.metrics import r2_score, mean_squared_error, make_scorer, mean_absolute_percentage_error
from sklearn.model_selection import LeavePOut, KFold, cross_val_score
from mapie.regression import MapieRegressor
import pandas as pd
from sklearn.pipeline import Pipeline


def get_k_scoring(col_count):
    """
    Get scores for all scoring metrics
    :param col_count:
    :return:
    """
    def adjusted_r2_k(y_true, y_pred, **kwargs):
        """
        Calculates adjusted R squared
        :param y_true:
        :param y_pred:
        :param kwargs:
        :return:
        """
        return 1 - (1 - r2_score(y_true, y_pred)) * (len(y_pred) - 1) / (len(y_pred) - col_count - 1)

    def root_mean_squared_error(y_true, y_pred, **kwargs):
        """
        Calculates root means squared error
        :param y_true:
        :param y_pred:
        :param kwargs:
        :return:
        """
        return np.sqrt(mean_squared_error(y_true, y_pred))

    def r_square(y_true, y_pred, **kwargs):
        """
        Calculates R squared error
        :param y_true:
        :param y_pred:
        :param kwargs:
        :return:
        """
        return r2_score(y_true, y_pred)

    adjusted_r_square_k = make_scorer(adjusted_r2_k, greater_is_better=True)
    r_square_k = make_scorer(r_square, greater_is_better=True)
    root_mean_squared_error_k = make_scorer(root_mean_squared_error, greater_is_better=False)
    mean_absolute_error_k = make_scorer(root_mean_squared_error, greater_is_better=False)
    mean_absolute_percentage_error_k = make_scorer(mean_absolute_percentage_error, greater_is_better=False)
    return {'Adjusted R-Squared': adjusted_r_square_k, 'R-Squared': r_square_k, 'RMSE': root_mean_squared_error_k,
            'MAE': mean_absolute_error_k, 'MAPE': mean_absolute_percentage_error_k}


def run_validation_leave_p_out(pipe, input_data, scores, y_pred):
    """
    Runs lave p out  cross validation
    :param pipe:
    :param input_data:
    :param scores:
    :param y_pred:
    :return:
    """
    cv = LeavePOut(p=input_data.user_param['validation']['leave_p_out']['p'])
    return run_cv(pipe, input_data, scores, cv)


def run_validation_kfold(pipe, input_data, scores, y_pred):
    """
    Runs k fold  cross validation
    :param pipe:
    :param input_data:
    :param scores:
    :param y_pred:
    :return:
    """
    cv = KFold(n_splits=input_data.user_param['validation']['kfold']['k'], shuffle=True, random_state=1)
    return run_cv(pipe, input_data, scores, cv)


def run_cv(pipe, input_data, scores, cv):
    """
    Generic cross validation function
    :param pipe:
    :param input_data:
    :param scores:
    :param cv:
    :return:
    """
    for scoring_method in input_data.user_param['scoring']:
        sc = get_k_scoring(input_data.X.shape[1])[scoring_method]
        cv_score = np.mean(cross_val_score(pipe, input_data.X, input_data.y, cv=cv, scoring=sc))
        weights = get_model_weights(pipe)
        pen_cv_score = cv_score + 0.1 * weights

        if scoring_method not in scores:
            scores[scoring_method] = []
        scores[scoring_method].append(cv_score)
        if input_data.user_param['penalized']:
            scores['penalized' + str(-scoring_method)].append(pen_cv_score)
    return scores


# Adjusted R-square function
def adjusted_r_squared(r2, n, p):
    """
    Adjusted R-square function
    :param r2:
    :param n:
    :param p:
    :return:
    """
    return 1 - (1 - r2) * ((n - 1) / (n - p - 1))


def get_score_r2(input_data, y_pred):
    """
    Helper function for R squared
    :param input_data:
    :param y_pred:
    :return:
    """
    return r2_score(input_data.y_test, y_pred)


def get_score_adjusted_r2(input_data, y_pred):
    """
     Helper function for adjusted R squared
    :param input_data:
    :param y_pred:
    :return:
    """
    r_squared = get_score_r2(input_data, y_pred)
    return adjusted_r_squared(r_squared, input_data.X_test.shape[0], input_data.X_test.shape[1])


def get_score_rmse(input_data, y_pred):
    """
    Helper function for RMSE
    :param input_data:
    :param y_pred:
    :return:
    """
    return np.sqrt(mean_squared_error(input_data.y_test, y_pred))


def get_score_mae(input_data, y_pred):
    """
    Helper function for MAE
    :param input_data:
    :param y_pred:
    :return:
    """
    return mean_squared_error(input_data.y_test, y_pred)


def get_score_mape(input_data, y_pred):
    """
    Helper function for MAPE
    :param input_data:
    :param y_pred:
    :return:
    """
    return mean_absolute_percentage_error(input_data.y_test, y_pred)


def run_validation_user_defined(pipe, input_data, scores, y_pred):
    """
    Run user defined validation
    :param pipe:
    :param input_data:
    :param scores:
    :param y_pred:
    :return:
    """
    return run_validation_standard(pipe, input_data, scores, y_pred)


def run_validation_standard(pipe, input_data, scores, y_pred):
    """
    Run standard validation
    :param pipe:
    :param input_data:
    :param scores:
    :param y_pred:
    :return:
    """
    name_mapping = {
        "Adjusted R-Squared": "adjusted_r2",
        "R-Squared": "r2",
        "RMSE": "rmse",
        "MAE": "mae",
        "MAPE": "mape"
    }

    penalize = ["RMSE", "MAE", "MAPE"]
    weights = get_model_weights(pipe)
    if "weights" not in scores:
        scores["weights"] = []
    scores["weights"].append(weights)

    for scoring_method in input_data.user_param['scoring']:
        if scoring_method not in scores:
            scores[scoring_method] = []
        score = eval('get_score_' + name_mapping[scoring_method] + '(input_data, y_pred)')
        scores[scoring_method].append(score)

        if input_data.user_param['penalized'] and scoring_method in penalize:
            if "penalized-" + scoring_method not in scores:
                scores["penalized-" + scoring_method] = []
            p_score = score + input_data.user_param['penalized_parameter'] * weights
            scores["penalized-" + scoring_method].append(p_score)
    return scores


def get_model_weights(pipe):
    """
    Retrieve weights from the regressors
    :param pipe:
    :return:
    """
    try:
        weights_n = pipe.named_steps['regressor'].n_estimators
    except AttributeError:
        weights_n = 0
    try:
        weights_c = len(pipe.named_steps['regressor'].coef_)
    except AttributeError:
        weights_c = 0
    return max(weights_n, weights_c)


def run_uncertainty(regress, input_data):
    """
    Run uncertainty calculations
    :param regress:
    :param input_data:
    :return:
    """
    name, model = regress
    params = input_data.get_parameters(name, model().get_params().keys())
    pipe = Pipeline(steps=[
        ("preprocessor", input_data.get_preprocessor()),
        ("regressor", model(**params))])
    alpha = 1 - input_data.user_param['uncertainty_calculation']['prediction_intervals']
    mapie = MapieRegressor(pipe, method=input_data.user_param['uncertainty_calculation']['method'])
    mapie.fit(input_data.X, input_data.y)
    y_preds, y_pis = mapie.predict(input_data.X, alpha=alpha)
    y_pis = y_pis.reshape(y_pis.shape[0], -1)
    uncern = np.concatenate((y_preds[:, None], y_pis), axis=1)
    uncern = pd.DataFrame(uncern, columns=['Prediction', 'Lower_bound', 'Upper_bound'])
    uncern['Range'] = uncern.apply(lambda x: x['Upper_bound'] - x['Lower_bound'], axis=1)
    t = input_data.X
    return uncern, input_data.user_param['uncertainty_calculation']['prediction_intervals']
